import tkinter as tk
from tkinter import ttk, filedialog, scrolledtext, messagebox
import threading
import time
import os
import json

class IntentionRepeaterGUI(tk.Tk):
    def __init__(self):
        super().__init__()

        self.title("Intention Repeater GUI Basic")
        self.geometry("800x900")

        self.running = False
        self.iterations = 0
        self.start_time = 0

        self.subject_image_path = None
        self.witness_image_path = None
        self.config_file = 'repeater_config.json'

        self.dark_mode_var = tk.BooleanVar()
        self.style = ttk.Style(self)
        
        self.create_widgets()
        self.load_config()

        self.update_idletasks()
        width = self.winfo_width()
        height = self.winfo_height()
        x = (self.winfo_screenwidth() // 2) - (width // 2)
        y = (self.winfo_screenheight() // 2) - (height // 2)
        self.geometry(f'{width}x{height}+{x}+{y}')


    def create_widgets(self):
        main_frame = ttk.Frame(self, padding="10")
        main_frame.pack(fill=tk.BOTH, expand=True)

        # Intention Frame
        intention_frame = ttk.LabelFrame(main_frame, text="Step 1: Define the Intention")
        intention_frame.pack(fill=tk.X, pady=5)
        
        description_label = ttk.Label(intention_frame, text="This is the core message or energy to be repeated.", font=("Helvetica", 9, "italic"))
        description_label.pack(anchor=tk.W, padx=5, pady=(2, 5))

        self.intention_source = tk.StringVar(value="text")

        self.intention_rb_text = ttk.Radiobutton(intention_frame, text="Intention Textbox", variable=self.intention_source, value="text")
        self.intention_rb_text.pack(anchor=tk.W, padx=5)
        self.intention_text = scrolledtext.ScrolledText(intention_frame, wrap=tk.WORD, width=60, height=10)
        self.intention_text.pack(fill=tk.BOTH, expand=True, padx=5, pady=5)

        self.intention_rb_file = ttk.Radiobutton(intention_frame, text="File", variable=self.intention_source, value="file")
        self.intention_rb_file.pack(anchor=tk.W, padx=5)
        file_frame = ttk.Frame(intention_frame)
        file_frame.pack(fill=tk.X, padx=5, pady=5)
        self.file_path_entry = ttk.Entry(file_frame, width=50)
        self.file_path_entry.pack(side=tk.LEFT, fill=tk.X, expand=True)
        self.load_file_button = ttk.Button(file_frame, text="Load File", command=self.load_file)
        self.load_file_button.pack(side=tk.LEFT, padx=5)

        # Burst Count Frame
        burst_frame = ttk.LabelFrame(main_frame, text="Burst Count")
        burst_frame.pack(fill=tk.X, pady=5)
        self.burst_count = tk.StringVar(value="1")
        self.burst_count_entry = ttk.Entry(burst_frame, textvariable=self.burst_count, width=10)
        self.burst_count_entry.pack(padx=5, pady=5)

        # --- MODIFIED SECTION START ---

        # Speed Frame
        speed_frame = ttk.LabelFrame(main_frame, text="Repetition Speed")
        speed_frame.pack(fill=tk.X, pady=5, ipady=5)
        
        # Configure grid to make the center column expand, pushing the dark mode controls to the right
        speed_frame.columnconfigure(1, weight=1) 
        
        self.speed = tk.StringVar(value="max")
        
        # Row 0: Maximum Speed radiobutton
        self.speed_rb_max = ttk.Radiobutton(speed_frame, text="Maximum Speed", variable=self.speed, value="max")
        self.speed_rb_max.grid(row=0, column=0, sticky='w', padx=5, pady=2)

        # Create a frame to hold the dark mode label and checkbox together on the right
        dark_mode_frame = ttk.Frame(speed_frame)
        dark_mode_frame.grid(row=0, column=2, sticky='e', padx=10)

        # Add the text label and the checkbox without text inside the frame
        self.dark_mode_label = ttk.Label(dark_mode_frame, text="Dark Mode")
        self.dark_mode_label.pack(side=tk.LEFT, anchor='e', padx=(0,5))
        self.dark_mode_checkbox = ttk.Checkbutton(dark_mode_frame, variable=self.dark_mode_var, command=self.toggle_dark_mode)
        self.dark_mode_checkbox.pack(side=tk.LEFT, anchor='w')

        # Row 1: Once-Per-Hour radiobutton
        self.speed_rb_hour = ttk.Radiobutton(speed_frame, text="Once-Per-Hour", variable=self.speed, value="hour")
        self.speed_rb_hour.grid(row=1, column=0, sticky='w', padx=5, pady=2)
        
        # --- MODIFIED SECTION END ---

        # Subject and Witness Frame
        subject_witness_frame = ttk.Frame(main_frame)
        subject_witness_frame.pack(fill=tk.X, pady=5)

        # Subject Frame
        subject_frame = ttk.LabelFrame(subject_witness_frame, text="Subject (Optional Target)")
        subject_frame.pack(side=tk.LEFT, fill=tk.BOTH, expand=True, padx=(0, 5))
        self.subject_type = tk.StringVar(value="text")
        self.subject_rb_text = ttk.Radiobutton(subject_frame, text="Text", variable=self.subject_type, value="text")
        self.subject_rb_text.pack(anchor=tk.W, padx=5)
        self.subject_text = ttk.Entry(subject_frame, width=30)
        self.subject_text.pack(fill=tk.X, padx=5, pady=5)
        self.subject_rb_image = ttk.Radiobutton(subject_frame, text="Image", variable=self.subject_type, value="image")
        self.subject_rb_image.pack(anchor=tk.W, padx=5)
        self.subject_image_label = ttk.Label(subject_frame, text="No image loaded", wraplength=150)
        self.subject_image_label.pack(fill=tk.X, padx=5, pady=5)
        self.subject_load_button = ttk.Button(subject_frame, text="Load Image", command=lambda: self.load_image('subject'))
        self.subject_load_button.pack(pady=5)

        # Witness Frame
        witness_frame = ttk.LabelFrame(subject_witness_frame, text="Witness (Optional Link)")
        witness_frame.pack(side=tk.LEFT, fill=tk.BOTH, expand=True, padx=(5, 0))
        self.witness_type = tk.StringVar(value="text")
        self.witness_rb_text = ttk.Radiobutton(witness_frame, text="Text", variable=self.witness_type, value="text")
        self.witness_rb_text.pack(anchor=tk.W, padx=5)
        self.witness_text = ttk.Entry(witness_frame, width=30)
        self.witness_text.pack(fill=tk.X, padx=5, pady=5)
        self.witness_rb_image = ttk.Radiobutton(witness_frame, text="Image", variable=self.witness_type, value="image")
        self.witness_rb_image.pack(anchor=tk.W, padx=5)
        self.witness_image_label = ttk.Label(witness_frame, text="No image loaded", wraplength=150)
        self.witness_image_label.pack(fill=tk.X, padx=5, pady=5)
        self.witness_load_button = ttk.Button(witness_frame, text="Load Image", command=lambda: self.load_image('witness'))
        self.witness_load_button.pack(pady=5)

        # Status Frame
        status_frame = ttk.LabelFrame(main_frame, text="Status")
        status_frame.pack(fill=tk.X, pady=5, ipady=5)
        self.elapsed_time_label = ttk.Label(status_frame, text="Elapsed Time: 00:00:00", font=("Helvetica", 24, "bold"))
        self.elapsed_time_label.pack(pady=10)
        self.iterations_label = ttk.Label(status_frame, text="Iterations: 0", font=("Helvetica", 18))
        self.iterations_label.pack(pady=5)
        self.status_label = ttk.Label(status_frame, text="Stopped", font=("Helvetica", 12))
        self.status_label.pack(pady=5)

        # Control Frame
        control_frame = ttk.Frame(main_frame)
        control_frame.pack(fill=tk.X, pady=10)
        self.start_button = ttk.Button(control_frame, text="Start", command=self.start_repeater)
        self.start_button.pack(side=tk.LEFT, fill=tk.X, expand=True, padx=(0, 5))
        self.stop_button = ttk.Button(control_frame, text="Stop", command=self.stop_repeater, state='disabled')
        self.stop_button.pack(side=tk.LEFT, fill=tk.X, expand=True, padx=(5, 0))

    def toggle_dark_mode(self):
        light_theme = {
            "root_bg": "#F0F0F0", "frame_bg": "#F0F0F0", "label_fg": "#000000",
            "entry_bg": "#FFFFFF", "entry_fg": "#000000", "button_bg": "#E1E1E1",
            "button_fg": "#000000", "active_button_bg": "#E5F1FB",
        }
        dark_theme = {
            "root_bg": "#2E2E2E", "frame_bg": "#2E2E2E", "label_fg": "#FFFFFF",
            "entry_bg": "#3C3C3C", "entry_fg": "#FFFFFF", "button_bg": "#505050",
            "button_fg": "#FFFFFF", "active_button_bg": "#646464",
        }
        
        colors = dark_theme if self.dark_mode_var.get() else light_theme
        
        self.style.theme_use('clam')
        self.config(background=colors["root_bg"])

        self.style.configure('.', background=colors["frame_bg"], foreground=colors["label_fg"])
        self.style.configure('TFrame', background=colors["frame_bg"])
        self.style.configure('TLabel', background=colors["frame_bg"], foreground=colors["label_fg"])
        self.style.configure('TLabelframe', background=colors["frame_bg"], bordercolor=colors["label_fg"])
        self.style.configure('TLabelframe.Label', background=colors["frame_bg"], foreground=colors["label_fg"])
        self.style.configure('TRadiobutton', background=colors["frame_bg"], foreground=colors["label_fg"])
        self.style.configure('TCheckbutton', background=colors["frame_bg"], foreground=colors["label_fg"])
        self.style.configure('TEntry', fieldbackground=colors["entry_bg"], foreground=colors["entry_fg"], insertcolor=colors["entry_fg"])
        self.style.configure('TButton', background=colors["button_bg"], foreground=colors["button_fg"], borderwidth=1)
        self.style.map('TButton', background=[('active', colors["active_button_bg"])])

        self.intention_text.config(background=colors["entry_bg"], foreground=colors["entry_fg"], insertbackground=colors["entry_fg"])

    def save_config(self):
        config = {
            'intention_source': self.intention_source.get(),
            'intention_text': self.intention_text.get('1.0', tk.END),
            'file_path': self.file_path_entry.get(),
            'burst_count': self.burst_count.get(),
            'speed': self.speed.get(),
            'subject_type': self.subject_type.get(),
            'subject_text': self.subject_text.get(),
            'subject_image_path': self.subject_image_path,
            'witness_type': self.witness_type.get(),
            'witness_text': self.witness_text.get(),
            'witness_image_path': self.witness_image_path,
            'dark_mode': self.dark_mode_var.get()
        }
        try:
            with open(self.config_file, 'w') as f:
                json.dump(config, f, indent=4)
        except Exception as e:
            print(f"Error saving config: {e}")

    def load_config(self):
        try:
            with open(self.config_file, 'r') as f:
                config = json.load(f)

            self.intention_source.set(config.get('intention_source', 'text'))
            self.intention_text.delete('1.0', tk.END)
            self.intention_text.insert('1.0', config.get('intention_text', ''))
            self.file_path_entry.delete(0, tk.END)
            self.file_path_entry.insert(0, config.get('file_path', ''))
            self.burst_count.set(config.get('burst_count', '1'))
            self.speed.set(config.get('speed', 'max'))
            self.subject_type.set(config.get('subject_type', 'text'))
            self.subject_text.delete(0, tk.END)
            self.subject_text.insert(0, config.get('subject_text', ''))
            self.witness_type.set(config.get('witness_type', 'text'))
            self.witness_text.delete(0, tk.END)
            self.witness_text.insert(0, config.get('witness_text', ''))

            s_path = config.get('subject_image_path')
            if s_path and os.path.exists(s_path):
                self.subject_image_path = s_path
                self.subject_image_label.config(text=os.path.basename(s_path))
            
            w_path = config.get('witness_image_path')
            if w_path and os.path.exists(w_path):
                self.witness_image_path = w_path
                self.witness_image_label.config(text=os.path.basename(w_path))

            self.dark_mode_var.set(config.get('dark_mode', False))
            self.toggle_dark_mode()

        except FileNotFoundError:
            self.toggle_dark_mode()
        except Exception as e:
            print(f"Error loading config: {e}")
            messagebox.showerror("Config Error", f"Could not load configuration file:\n{e}")

    def load_file(self):
        filepath = filedialog.askopenfilename(
            title="Open Intention File",
            filetypes=(("Text files", "*.txt"), ("All files", "*.*"))
        )
        if filepath:
            self.file_path_entry.delete(0, tk.END)
            self.file_path_entry.insert(0, filepath)
            self.intention_source.set("file")

    def load_image(self, image_type):
        filepath = filedialog.askopenfilename(
            title=f"Load {image_type.capitalize()} Image",
            filetypes=(("Image files", "*.png *.jpg *.jpeg *.gif *.bmp"),)
        )
        if filepath:
            if image_type == 'subject':
                self.subject_image_path = filepath
                self.subject_image_label.config(text=os.path.basename(filepath))
                self.subject_type.set("image")
            elif image_type == 'witness':
                self.witness_image_path = filepath
                self.witness_image_label.config(text=os.path.basename(filepath))
                self.witness_type.set("image")

    def _set_controls_state(self, state):
        text_state = tk.NORMAL if state == 'normal' else tk.DISABLED
        self.intention_text.config(state=text_state)
        self.intention_rb_text.config(state=state)
        self.intention_rb_file.config(state=state)
        self.file_path_entry.config(state=state)
        self.load_file_button.config(state=state)
        self.burst_count_entry.config(state=state)
        self.speed_rb_max.config(state=state)
        self.speed_rb_hour.config(state=state)
        self.subject_rb_text.config(state=state)
        self.subject_text.config(state=state)
        self.subject_rb_image.config(state=state)
        self.subject_load_button.config(state=state)
        self.witness_rb_text.config(state=state)
        self.witness_text.config(state=state)
        self.witness_rb_image.config(state=state)
        self.witness_load_button.config(state=state)
        self.dark_mode_checkbox.config(state=state)

    def start_repeater(self):
        self.save_config()
        self.running = True
        self.iterations = 0
        self.start_time = time.time()
        self.start_button.config(state='disabled')
        self.stop_button.config(state='normal')
        self._set_controls_state('disabled')
        self.status_label.config(text="Running...")

        self.repeater_thread = threading.Thread(target=self.repeater_loop, daemon=True)
        self.repeater_thread.start()
        self.update_status()

    def stop_repeater(self):
        self.running = False
        self.start_button.config(state='normal')
        self.stop_button.config(state='disabled')
        self._set_controls_state('normal')
        self.status_label.config(text="Stopped")

    def repeater_loop(self):
        while self.running:
            try:
                burst_count = int(self.burst_count.get())
            except ValueError:
                burst_count = 1

            intention = ""
            if self.intention_source.get() == "text":
                intention = self.intention_text.get("1.0", tk.END).strip()
            elif self.intention_source.get() == "file":
                try:
                    with open(self.file_path_entry.get(), 'r', encoding='utf-8') as f:
                        intention = f.read()
                except Exception as e:
                    self.after(0, lambda: messagebox.showerror("File Error", f"Could not read file: {e}"))
                    self.after(0, self.stop_repeater)
                    return

            if not intention:
                self.after(0, lambda: messagebox.showwarning("Input Error", "The Intention source is empty."))
                self.after(0, self.stop_repeater)
                return

            for _ in range(burst_count):
                if not self.running:
                    break
                _ = intention
                self.iterations += 1

            if self.speed.get() == "hour" and self.running:
                self.after(0, lambda: self.status_label.config(text="Sleeping for one hour..."))
                time.sleep(3600)
                if self.running:
                    self.after(0, lambda: self.status_label.config(text="Running..."))

    def update_status(self):
        if self.running:
            elapsed_seconds = int(time.time() - self.start_time)
            hours, remainder = divmod(elapsed_seconds, 3600)
            minutes, seconds = divmod(remainder, 60)
            self.elapsed_time_label.config(text=f"Elapsed Time: {hours:02}:{minutes:02}:{seconds:02}")
            self.iterations_label.config(text=f"Iterations: {self.format_iterations(self.iterations)}")
            self.after(1000, self.update_status)

    def format_iterations(self, n):
        if n < 1000:
            return str(n)
        for unit in ['', 'k', 'M', 'B', 'T', 'Q']:
            if abs(n) < 1000.0:
                return f"{n:3.1f}{unit}"
            n /= 1000.0
        return f"{n:.1f}Q"

if __name__ == "__main__":
    app = IntentionRepeaterGUI()
    app.mainloop()